/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

 /* ---------------------------------------------------------------
 * File: xexer.c 
 *       general Exerciser functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xiocommo.h>

#include <xpciapi.h>
#include <xaddrmap.h>

#include <xregcons.h>
#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>

/* Pointer to exerciser database */
#define DB_EXERCISER (bx_handlearray[handle].db->Exerciser)

/* Value of generic property */
#define GENERIC(prop) (DB_EXERCISER.GenProp[prop])

#define RIBLK_UPD (&DB_EXERCISER.RequesterInitiator.Block.UpdateInfo)
#define RIBEH_UPD (&DB_EXERCISER.RequesterInitiator.Behavior.UpdateInfo)
#define CTBEH_UPD (&DB_EXERCISER.CompleterTarget.Behavior.UpdateInfo)
#define CIBEH_UPD (&DB_EXERCISER.CompleterInitiator.Behavior.UpdateInfo)
#define RTBEH_UPD (&DB_EXERCISER.RequesterTarget.Behavior.UpdateInfo)

/********************************************************************
  Exerciser Run Control *********************************************
********************************************************************/

/*---------------------------------------------------------------------------*
 * BestXExerciserRun()
 *
 * Purpose:  Resets the exerciser and calls BestXExerciserContinue().
 *           Resetting is done by switching to prog-mode and back again.
 *           Precondition: This function may only be called in run-mode.
 *           Returns an error if one of the following conditions is true:
 *           - exerciser running (includes waiting for conditional start)
 *           - completer queues not empty
 *           - not all tags free
 *           - master enable bit not set
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserRun(
  bx_handletype  handle)
{
  bx_int32 progmode=0;
  bx_int32 pcixmode=0;

  BX_DECLARE_FUNCNAME("BestXExerciserRun [erun]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* just to be really sure check it */
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */ 

    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_EXERCISER));
    
    BX_TRY(BestXExerciserIsInProgMode(handle,&progmode));
    BX_E_ERROR_MSG_SET("BestXExerciserRun: Fatal error: Card is in programming mode. Please close and reopen connection");
    BX_TRY_FAIL(progmode?BX_E_ERROR:BX_E_OK);

    /* Check, wether card is in PCI-X mode */
    BX_TRY(BestXStatusRead(handle,BX_STAT_PCIXMODE,&pcixmode));
    if (!pcixmode)
    {
      BX_E_ERROR_MSG_SET("BestXExerciserRun: Card cannot run in PCI mode");
      BX_TRY_ERROR(BX_E_ERROR);
    }
    
    /* Clear compare error */
    BX_TRY(BestXStatusClear(handle,BX_STAT_CMP_FAIL));

#if 0 /* if you uncomment this, be sure to update the doku too (chris) !! */
    /* Clear SERR,PERR error */
    BX_TRY(BestXStatusClear(handle,BX_STAT_ERR_SERR));
    BX_TRY(BestXStatusClear(handle,BX_STAT_ERR_PERR));

    /* Clear parity error */
    BX_TRY(BestXStatusClear(handle,BX_STAT_ERR_WRPAR)); /* clears WRPAR64 too */
#endif

    /* Clear split fail message bit */
    BX_TRY(BestXStatusClear(handle,BX_STAT_SPLITFAIL)); 
    
    /* Clear master and target abort (not resetted by PCIRST# !!) */
    BX_TRY(BestXStatusClear(handle,BX_STAT_IABORT)); /* clears TABORT too */
    
    /* Clear all exerciser statemachines and counters (pipelines)
       except for the bus statemachine. Completer is also cleared ! 
       This does a switch to prog-mode and back again !
    
       CAUTION: The call to ExerciserClear() has been removed from here
                and must be done by the caller before calling ExerciserRun(). 
                Exception: When ExerciserProg() has been executed 
                before, you dont need to call ExerciserClear() additionally.
                The problem lies in the fact that ExerciserClear() 
                (independent of its implementation) also clears
                the Completer, i.e. outstanding split completions
                would never be executed (other RT would wait forever).
                Scenario: A test with two cards (say A+B).
                          1. Test executes ExerciserRun() on card A 
                          2. A's RI executes and immediately gets a split from B's CT
                          3. Test calls ExerciserRun() on card B.
                             This would clear the outstanding split, i.e.
                             B's CI would never complete and A's RT would wait forever.
                Solution: The test now calls ExerciserClear() for A and B before
                          calling ExerciserRun() for A and B. Since ExerciserRun()
                          does not call ExerciserClear() any more, all clearing is
                          done before the first call to ExerciserRun().
    
       BX_TRY(BestXExerciserClear(handle)); do not uncomment this !!!
    */

    /* We may be in break state here, i.e the exerciser will not access the bus.
       Clear hardware-break
    */
    BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_RUN_CTRL_STATE_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_REG,
                             sizeof(bx_int16),
                             0x3, /* mask for Stop+Break */
                             0x0,
                             
                             BX_REG_EX_RUN_CTRL_SYNC_REG,
    
                             BX_REG_EX_RUN_CTRL_FORCE_REG,
                             sizeof(bx_int16),
                             1 /* do soft force */
                             ));


    /* Run the exerciser, this checks the master enable bit */
    BX_TRY(BestXExerciserContinue(handle));

    BX_TRY_PROGRESS(BestXResourceUnlock(handle,BX_RESLOCK_EXERCISER));
  }
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserStop()
 *
 * Purpose:  Stops the exerciser protocol-conform
 *           without data-loss.
 *           Finishes current sequence.
 *           Requester still accepts all outstanding
 *           requests, i.e. completer still running.
 *           ExerciserContinue() possible afterwards.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserStop(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserStop [estop]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_EXERCISER));

    
    /* if license is checked here, GUI won't run ... */
    /* BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); *//* EMR */
    /* We may not be in runmode here:
       This function is also called in progmode from BestXExerciserRunMode() !!

       During BestOpen(), PCIRST may be asserted or no clock available.
       So this function must succeed (-> Do soft force) 

    */

    BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_RUN_CTRL_STATE_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x0,
                             
                             BX_REG_EX_RUN_CTRL_SYNC_REG,
    
                             BX_REG_EX_RUN_CTRL_FORCE_REG,
                             sizeof(bx_int16),
                             1 /* force, if needed */
                             ));

    BX_TRY_PROGRESS(BestXResourceUnlock(handle,BX_RESLOCK_EXERCISER));
  }
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }
  /* We may not be stopped here !
     Caller needs to poll BX_STAT_TEST to find out.
  */
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserPause()
 *
 * Purpose:  Currently mapped to ExerciserStop() 
 *
 *           Stops the exerciser protocol-conform
 *           without data-loss.
 *           Finishes current sequence.
 *           Requester still accepts all outstanding
 *           requests, i.e. completer still running.
 *           ExerciserContinue() possible afterwards.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserPause(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserPause []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */
    BX_TRY(BestXExerciserStop(handle));
  }

  /* We may not be stopped here !
     Caller needs to poll BX_STAT_TEST to find out.
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserContinue()
 *
 * Purpose: Checks the master enable bit and starts the exerciser.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserContinue(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserContinue [econt]");
  bx_int32 progmode,val;

  BX_TRY_VARS;

  BX_TRY_BEGIN

  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */
    
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_EXERCISER)); 

    /* just to be really sure check it (chris: TBD remove this check later on) */
    BX_TRY(BestXExerciserIsInProgMode(handle,&progmode));
    BX_E_ERROR_MSG_SET("BestXExerciserContinue: Fatal error: Card is in programming mode. Please close and reopen connection")
    BX_TRY_FAIL(progmode?BX_E_ERROR:BX_E_OK);

    /* Check the master enable bit */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),&val)); 
    if ((val&(1<<2))==0)
    {
      /* Master enable bit not set */
      BX_E_ERROR_MSG_SET("ERROR BestXExerciser: Master enable bit not set")
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* Run the exerciser */
    BX_TRY(BestXSyncRegWrite(handle,
                             BX_REG_EX_RUN_CTRL_STATE_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_SYNC_REG,
    
                             BX_REG_EX_RUN_CTRL_FORCE_REG,
                             sizeof(bx_int16),
                             0 /* no soft force */
                             ));
    
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER)); 
  }
    BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserClear()
 *
 * Purpose: Clear all exerciser statemachines and counters (pipelines)
 *          except for the bus statemachine. Completer also cleared ! 
 *          Called from ExerciserRun(). Need to be in run-mode 
 *          and stopped before calling this functions.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserClear(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserClear [eclear]");

  bx_int32 progmode;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* The exerciser clear bit in the Exerciser Run Control Register is
       quite difficult to handle. Same behavior can be reached by 
       switching to prog-mode and back to runmode.
       CAUTION: SW avoids real HW switching to progmode, if we are 
       already in progmode.
       So this works as long as we are in runmode here,
       which should always be the case.
    */
    /* just to be really sure check it */
    BX_TRY(BestXExerciserIsInProgMode(handle,&progmode));
    BX_E_ERROR_MSG_SET("BestXExerciserClear: Fatal error: Card is in programming mode. Please close and reopen connection")
    BX_TRY_FAIL(progmode?BX_E_ERROR:BX_E_OK);


    /* If exerciser is still running, we get an error here.
       Caller should execute an ExerciserStop() before.
    */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */)); /* real HW-switch here !!! */
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* switch to progmode successful */
      BX_TRY_PASSED
      {
        /* switch to runmode successful */
      }
      BX_TRY_FAILED
      {
        /* switch to runmode not successful */
        (void)BestXExerciserRunMode(handle,1 /* soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserBreak()
 *
 * Purpose:
 *          Bring bus to idle as soon as possible
 *          in a protocol conform way. 
 *          Data-loss may occur, because outstanding 
 *          completer transactions or retries won't
 *          be performed. 
 *          Example: A target which has responded
 *                   with split, has crashed.
 *                   We would have to wait forever.
 *
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXExerciserBreak(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserBreak [ebreak]");
  
  bx_int32 progmode,val;
  bx_int32 NumTries=3; /* chris TBD make this a property */
  bx_int32 DecoderEnable;
  bx_int32 Dac0Enable,Dac2Enable,Dac4Enable;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */
    /* just to be really sure check it (chris: TBD remove this check later on) */
    
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_EXERCISER));  
   
    BX_TRY(BestXExerciserIsInProgMode(handle,&progmode));
    BX_E_ERROR_MSG_SET("BestXExerciserBreak: Fatal error: Card is in programming mode. Please close and reopen connection");
    BX_TRY_FAIL(progmode?BX_E_ERROR:BX_E_OK);

    /* Do a hardware-break (stop with next transaction) */
    BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_RUN_CTRL_STATE_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_REG,
                             sizeof(bx_int16),
                             0x3, /* mask for Stop+Break chris: TBD: check wether 0x2 is enough */
                             0x2,
                             
                             BX_REG_EX_RUN_CTRL_SYNC_REG,
    
                             BX_REG_EX_RUN_CTRL_FORCE_REG,
                             sizeof(bx_int16),
                             0 /* no soft force */
                             ));

    /* Wait for the RI statemachine becoming idle 
      (it may currently executing a long burst transaction
    */

    NumTries=3; /* chris TBD make this a property */
    do
    {
      assert(NumTries); /* zero would lead to an endless loop here */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_INFO_REG,sizeof(bx_int16),&val));
      if (val & 1<<1)
      {
        /* RI state machine idle */
        break; /* Ends the do-while() loop */
      }
    } while(--NumTries);

    /* If RI hangs, user should execute a Reset here */
    BX_E_ERROR_MSG_SET("BestXExerciserBreak: RI hangs, you might execute a Reset.");
    BX_TRY_FAIL(NumTries?BX_E_OK:BX_E_ERROR);

    /* Remember enabled decoders */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_EN_REG,sizeof(bx_int16),&DecoderEnable));

    /* Remember DAC enable bits */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_STD0_CTRL_REG,sizeof(bx_int16),&Dac0Enable));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_STD2_CTRL_REG,sizeof(bx_int16),&Dac2Enable));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_STD4_CTRL_REG,sizeof(bx_int16),&Dac4Enable));

    /* Switch off decoders, except config decoder.
       This register is only checked by card at the beginning of each new
       transaction, i.e. a currently running transaction will not be 
       affected. */
    BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_DEC_EN_STATUS_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_DEC_EN_REG,
                             sizeof(bx_int16),
                             0x1FF,
                             0x100, /* config on */
                             
                             BX_REG_DEC_EN_SREG,
    
                             BX_REG_DEC_EN_FORCE_REG,
                             sizeof(bx_int16),
                             0 /* no soft force */
                             ));
        
    /***********************************************************************************/
    /* HW bugfix: The DAC enable bits must be cleared too, when decoder is disabled !! */
    /***********************************************************************************/
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD0_CTRL_REG,sizeof(bx_int16),1<<6,0));
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD2_CTRL_REG,sizeof(bx_int16),1<<6,0));
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD4_CTRL_REG,sizeof(bx_int16),1<<6,0));
    /***********************************************************************************/

    /* Wait for the target statemachine becoming idle 
      (it may currently executing a long burst transaction)
    */

    NumTries=3; /* chris TBD make this a property */
    do
    {
      assert(NumTries); /* zero would lead to an endless loop here */

      /* Check, wether target is idle or not */ 
      /* Clear RT and CT active bits and read out again */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),1<<14|1<<13));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&val));

      if ((val & (1<<14|1<<13))==0)
      {
        /* Target state machine idle */
        break; /* Ends the do-while() loop */
      }
    } while(--NumTries);

    /* If target hangs, user should execute a Reset here */
    BX_E_ERROR_MSG_SET("BestXExerciserBreak: Target hangs, you might execute a Reset.");
    BX_TRY_FAIL(NumTries?BX_E_OK:BX_E_ERROR);

    /* Do the reset. This clears the break */
    BX_TRY(BestXExerciserReset(handle));  
  
    /* Enable decoders again */
    BX_TRY(BestXSyncRegWrite(handle,
                             
                           BX_REG_DEC_EN_STATUS_REG,
                           sizeof(bx_int16),
                           0x1,
                           0x1,

                           BX_REG_DEC_EN_REG,
                           sizeof(bx_int16),
                           0x1FF,
                           DecoderEnable,

                           BX_REG_DEC_EN_SREG,

                           BX_REG_DEC_EN_FORCE_REG,
                           sizeof(bx_int16),
                           0 /* no soft force */));


    /* Restore DAC enable bits to its previous state */
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD0_CTRL_REG,sizeof(bx_int16),1<<6,Dac0Enable));
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD2_CTRL_REG,sizeof(bx_int16),1<<6,Dac2Enable));
    BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD4_CTRL_REG,sizeof(bx_int16),1<<6,Dac4Enable));
    
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER));   
  }
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserReset()
 *
 * Purpose: 
 *          Initiator and target are stopped immediately.
 *          This may cause protocol errors and data-loss.
 *          Resets all exerciser state machines, block-
 *          and behavior counters.
 *          Static settings like DBI registers and 
 *          memories are not affected. IABORT, TABORT are cleared !
 *  
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserReset(
  bx_handletype  handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserReset []");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */
    /* If there was a previous ExerciserRun() before, the HW reset
       will lead to the exerciser running again.
       To avoid this, we first stop the exerciser.
    */
    /* we lock the resource exerciser */
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_EXERCISER));
    
    BX_TRY(BestXExerciserStop(handle));

    /* Write an arbitrary value to get an asynchronous reset */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_EX_RUN_CTRL_RESET,sizeof(bx_int16),0));

    /* Resetting the exerciser by writing to BX_REG_EX_RUN_CTRL_RESET 
       does not get the statemachine out of the Break-state, so
       a following ExerciserRun() will not work.
       Clear the Break:
    */

    /* Clear hardware-break */
    BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_RUN_CTRL_STATE_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_EX_RUN_CTRL_REG,
                             sizeof(bx_int16),
                             0x3, /* mask for Stop+Break */
                             0x0,
                             
                             BX_REG_EX_RUN_CTRL_SYNC_REG,
    
                             BX_REG_EX_RUN_CTRL_FORCE_REG,
                             sizeof(bx_int16),
                             1 /* soft force */
                             ));

    /* Clear master and target abort (not resetted by PCIRST# !!) */
    BX_TRY(BestXStatusClear(handle,BX_STAT_IABORT)); /* clears TABORT too */

    /* we unlock the resource exerciser */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER));
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/********************************************************************
  Exerciser programming *********************************************
********************************************************************/

/* Variable CardIsInProgrammingMode is used as a kind of semaphore, 
   in order to keep the actual number of switchings
   betweeen Run- and Prog- mode as small as possible.
   It is incremented for every call to BestXExerciserProgMode()
   and decremented for a call to BestXExerciserRunMode().
   When it is decremented to zero, we actually switch 
   the card into Run-mode. 
   When it is incremented to one, we actually switch 
   the card into Programming-mode.
   Calls to BestXExerciserProgMode() and BestXExerciserRunMode()
   should always be balanced !
   After a power cycle, the card starts up in Prog-mode.
*/

/*---------------------------------------------------------------------------*
 * BestXExerciserProgMode()
 *
 * Purpose: Switches the exerciser into programming mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserProgMode(bx_handletype handle, bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXExerciserProgMode [eprogmode]");

  bx_int32 val;

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (bx_handlearray[handle].CardIsInProgrammingMode)
    {
      /* We are already in Programming-mode,
         do nothing but incrementing the semaphor (see below).
      */
    }
    else
    {
      /* We are in RunMode, need to switch into Prog-mode. */

      /* Check, wether we are allowed to switch into programming mode */
      BX_TRY(BestXStatusRead(handle,BX_STAT_TEST,&val));

      if (!(BestXIsOffline(handle) || val & BX_STAT_TEST_STOPPED))
      {
        /* We are not offline here and are not in state STOPPED.
           A STOP, BREAK or RESET is needed before. */

        BestXLastErrorParamSet(handle,BX_ERRPAR_1,1 /* Master busy */);
        BX_TRY_FAIL(BX_EFW_EX_INIMODESET_FAILED);
      }      
      
      /* If there was a previous ExerciserRun() before, the 
         HW-switch to runmode or a HW-reset
         will lead to the exerciser running again.
         To avoid this, we first stop the exerciser
         (we actually may not need it _here_).
      */
      BX_TRY(BestXExerciserStop(handle));

      /* This hook is used by the OEM to perform any functions
         related to entering prog mode.    */
      #ifdef HOOK_BESTX_BEFORE_PROG_MODE
        HOOK_BESTX_BEFORE_PROG_MODE ;
      #endif
      
      /* Set force retry bit */
      BX_TRY(BestXSyncRegWrite(handle,
                           
                             BX_REG_FORCE_RETRY_BIT_REG, /* Status register */
                             sizeof(bx_int16),
                             0x2,  /* Mask for Bit 1    */
                             0x2,  /* 1 in Bit 1=Ready  */
                           
                             BX_REG_FORCE_RETRY_BIT_REG, /* Control register */
                             sizeof(bx_int16),
                             0x1, /* Mask for bit 0 */
                             1,   /* 1=Retry        */
                           
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG,  /* Sync register */
  
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG, /* Force register */
                             sizeof(bx_int16),
                             SoftForce
                             ));
      
      /* Exerciser has stopped or we are in oflline mode.
         Now actually switch to prog-mode.
      */
      
      /* Remark: 
         Writing to control register resets all block 
         and behavior counters to point to the first row !!!
      */

      /* Layout of sync-mechanism for exerciser:
         1. MEPHISTO_OFFSET+0x20: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x24: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_EX_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             0,   /* 0=ProgMode     */
                             
                             BX_REG_EX_CLK_SWITCH_STAT_REG,
    
                             BX_REG_EX_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             SoftForce
            ));
                              
    } /* else */

    /* We always increment, in order to keep correct balancing */
    bx_handlearray[handle].CardIsInProgrammingMode++;
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserRunMode()
 *
 * Purpose: Switches the exerciser into run mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserRunMode(bx_handletype handle, bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXExerciserRunMode [erunmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (bx_handlearray[handle].CardIsInProgrammingMode==1)
    {
      /* We actually have to do a real HW-switch in this case,
         if (>1) we only have to decrement
      */
      
      /* Remark:
         Writing to control register resets all block 
         and behavior counters to point to the first row !!!
      */

      /* Layout of sync-mechanism:
         1. MEPHISTO_OFFSET+0x20: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x24: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      /* If there was a previous ExerciserRun() before, the 
         HW-switch to runmode
         will lead to the exerciser running again.
         To avoid this, we first stop the exerciser.
      */
      BX_TRY(BestXExerciserStop(handle)); /* does a SoftForce ! */

      /* Switch to runmode */
      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_EX_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_EX_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             1,   /* 1=RunMode      */
                             
                             BX_REG_EX_CLK_SWITCH_STAT_REG,
    
                             BX_REG_EX_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             SoftForce
                             ));
                             
      /* Clear the force retry bit */
      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_FORCE_RETRY_BIT_REG, /* Status register */
                             sizeof(bx_int16),
                             0x2,  /* Mask for Bit 1    */
                             0x2,  /* 1 in Bit 1=Ready  */
                             
                             BX_REG_FORCE_RETRY_BIT_REG, /* Control register */
                             sizeof(bx_int16),
                             0x1, /* Mask for bit 0 */
                             0,   /* 0=no Retry        */
                             
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG,  /* Sync register */
    
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG, /* Force register */
                             sizeof(bx_int16),
                             SoftForce
                             ));

      /* This hook is used by the OEM to perform any functions
         related to entering run mode. */
      #ifdef HOOK_BESTX_AFTER_RUN_MODE
        HOOK_BESTX_AFTER_RUN_MODE ;
      #endif

    }
    else
    {
      assert(bx_handlearray[handle].CardIsInProgrammingMode!=1);
      /* Either we are already in Runmode here (do nothing) or
         we are in programming mode but do actually not need to
         switch, balancing is enough.
      */
    }

    if (bx_handlearray[handle].CardIsInProgrammingMode)
    {
      /* balancing */
      bx_handlearray[handle].CardIsInProgrammingMode--;
    }
    else
    {
      /* If we are already in RunMode, we ignore this call completely. */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserIsInProgMode()
 *
 * Purpose: Checks, wether CAPI thinks that card is in prog-mode or not 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserIsInProgMode(bx_handletype handle,
                                             bx_int32 *ProgMode)
{
  BX_DECLARE_FUNCNAME("BestXExerciserIsInProgMode [eisinprogmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (ProgMode)
    {
      *ProgMode=bx_handlearray[handle].CardIsInProgrammingMode;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserProg()
 *
 * Purpose:  Writes complete exerciser settings to card 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXExerciserProg(
  bx_handletype  handle
)
{
  BX_DECLARE_FUNCNAME("BestXExerciserProg [eprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER); /* EMR */

    /* We lock the resource exerciser */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_EXERCISER));
    
    /* Switch to Prog-Mode (DBI clock).
       This is also done in each of the following Prog() functions.
       But this additional call makes sure that the actual 
       HW-switch is only done once and not in each function,
       in order to save time.
    */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
        
    /* 
    Remark:
    The following Prog() functions call
    SetUpdateInfo() after writing
    to card.
    */
    BX_TRY(BestXExerciserGenProg(handle));   /* program exerciser */
    BX_TRY(BestXRIProg(handle));             /* program RI */
    BX_TRY(BestXCTProg(handle));             /* program CT */
    BX_TRY(BestXCIProg(handle));             /* program CI */
    BX_TRY(BestXRTProg(handle));             /* program RT */

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  
    /* we unlock the resource exerciser */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER));
  }


  /* Error-handling */
  BX_TRY_CATCH
  {
    /* Error occurred somewhere */
    BX_TRY_PASSED
    {
      /* ResourceLock() returned BX_E_OK */
      BX_TRY_PASSED
      {
        /* ExerciserProgMode() returned BX_E_OK */
        BX_TRY_PASSED
        {
          /* all Prog() functions returned BX_E_OK */
          BX_TRY_PASSED
          {
            /* ExerciserRunMode() returned BX_E_OK */
            BX_TRY_PASSED
            {
              /* ResourceUnlock() returned BX_E_OK */
            }
            BX_TRY_FAILED
            {
              /* ResourceUnlock returned an error */
              (void)BestXAllResourceUnlock(handle);
            }
          }
          BX_TRY_FAILED
          {
            /* ExerciserRunMode() returned an error, do a force, but keep error: */
            (void)BestXExerciserRunMode(handle,1 /* soft force */);
            (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
          }
        }
        BX_TRY_FAILED
        {
          /* One of the Prog() functions returned an error */
          /* Switch back to Run-Mode (PCI clock) */
          (void)BestXExerciserRunMode(handle,1 /* soft force */);
          /* we unlock the resource exerciser */
          (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
        }
      }
      BX_TRY_FAILED
      {
        /* ExerciserProgMode() returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* ResourceLock() returned an error */
    }
  } /* CATCH */

  BX_ERRETURN(BX_TRY_RET);
}  


/*---------------------------------------------------------------------------*
 * BestXExerciserRead()
 *
 * Purpose:  Reads complete exerciser settings from card 
 *---------------------------------------------------------------------------*/
  
bx_errtype EXPORT BestXExerciserRead(
  bx_handletype  handle,
  bx_int32 option /* wether and what to read from Beh- and Block- memories */
)
{
  BX_DECLARE_FUNCNAME("BestXExerciserRead [eread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* we lock the resource exerciser*/
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_EXERCISER));
    /* 
    Remark:
    The following functions call
    SetUpdateInfo() after reading
    from card.
    */

    BX_TRY(BestXExerciserGenRead(handle));
    BX_TRY(BestXRIRead(handle,option));
    BX_TRY(BestXCTRead(handle,option));
    BX_TRY(BestXCIRead(handle,option));
    BX_TRY(BestXRTRead(handle,option));

    /* we unlock the resource exerciser */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXResourceUnlock(handle,BX_RESLOCK_EXERCISER));
  }

  BX_TRY_CATCH
  {
    /* Error occured */
    BX_TRY_PASSED
    {
      /* Resource locking was successful */
      BX_TRY_PASSED
      {
        /* all the Read-functions were successful */
        BX_TRY_PASSED
        {
          /* Resource unlocking was successful */
        }
        BX_TRY_FAILED
        {
          /* Resource unlocking was not successful */
          (void) BestXAllResourceUnlock(handle);
        }
      }
      BX_TRY_FAILED
      {
        /* one of the Read-functions returned an error */
        (void)BestXResourceUnlock(handle,BX_RESLOCK_EXERCISER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking was not successful */
    }
  } /* CATCH */

  BX_ERRETURN(BX_TRY_RET);
}  


/********************************************************************
  Exerciser generics ************************************************
********************************************************************/

bx_errtype EXPORT BestXExerciserGenGet(
  bx_handletype handle,
  bx_egentype prop,
  bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXExerciserGenGet [egenget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_EGEN_GET_BEGIN
      HOOK_BESTX_EGEN_GET_BEGIN ;
    #else

    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_EXERCISER_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    /* get property */
    *val=(bx_int32)GENERIC(prop);
    #endif
  
  }

  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXExerciserGenSet(
  bx_handletype handle,
  bx_egentype prop,
  bx_int32 val)

{
  BX_DECLARE_FUNCNAME("BestXExerciserGenSet [egenset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
  
    #ifdef HOOK_BESTX_EGEN_SET_BEGIN
      HOOK_BESTX_EGEN_SET_BEGIN ;
    #else
    
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_EXERCISER_GEN, (bx_int32)prop, val));

    /* set property */
    GENERIC(prop)=val;
    #endif
  }

  BX_ERRETURN(BX_TRY_RET);
}



bx_errtype EXPORT BestXExerciserGenDefaultSet(
  bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserGenDefaultSet [egendefset]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_EXERCISER_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_EXERCISER_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXExerciserGenSet(handle,
              ParamInfo->proptyp.egenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserGenProg
 *
 * Purpose: Writes all generic properties to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXExerciserGenProg(bx_handletype  handle)
{
  /* Updates Generics memory on card */

  BX_DECLARE_FUNCNAME("BestXExerciserGenProg [egenprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 val1,val2,val3,val4,val5,val6;

    #ifdef HOOK_BESTX_EGEN_PROG_BEGIN
      HOOK_BESTX_EGEN_PROG_BEGIN ;
    #endif

    /* Switch to prog mode */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* BX_EGEN_COMPLIANT */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_COMPLIANT,&val1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_PCI_X_COMPLIANCE_REG,sizeof(bx_int16),val1));

    /* BX_EGEN_ERR_SOURCE and BX_EGEN_ERR_NUM */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_SOURCE,&val1));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_NUM,&val2));

    switch (val1)
    {
      case BX_EGEN_ERR_SOURCE_NONE:
        /* No error generation at all */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),0));
        /* BX_EGEN_ERR_NUM irrelevant in this case */
        break;
      case BX_EGEN_ERR_SOURCE_RIBLK:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_RIGEN_NUMBLK[
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error block is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBLK,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_RIGEN_NUMBLK");
        
        #ifdef HOOK_BESTX_EGEN_PROG_RI_ERR_NUM
          HOOK_BESTX_EGEN_PROG_RI_ERR_NUM ;
        #endif
        
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_ERR_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_ERR_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable Block errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<0));
        break;
      case BX_EGEN_ERR_SOURCE_RIBEH:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_RIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_RIGEN_NUMBEH");

        #ifdef HOOK_BESTX_EGEN_PROG_RI_ERR_NUM
          HOOK_BESTX_EGEN_PROG_RI_ERR_NUM ;
        #endif

        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_ERR_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEH_ERR_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable master behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<1));
        break;
      case BX_EGEN_ERR_SOURCE_CTBEH:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_CTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCTGenGet (handle,BX_CTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_CTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_ERR_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEH_ERR_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable target behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<4));
        break;
      case BX_EGEN_ERR_SOURCE_CIBEH:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_CIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCIGenGet (handle,BX_CIGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_CIGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_ERR_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEH_ERR_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable completer behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<2));
        break;
      case BX_EGEN_ERR_SOURCE_RTBEH:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_RTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRTGenGet (handle,BX_RTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_RTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_ERR_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEH_ERR_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable requester behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<3));
        break;
      case BX_EGEN_ERR_SOURCE_DEC:
        /*
           If val2 (BX_EGEN_ERR_NUM) not in [0,..,BX_DECODER_SIZE[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           decoder is never accessed (because it does not exist).
        */ 
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_ERR_NUM should be less than BX_DECODER_SIZE");
        BX_TRY_FAIL((val2<BX_DECODER_SIZE ? BX_E_OK:BX_E_ERROR));
        
        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val2==BX_DEC_RT /* ==7 */)
        {
          /* Source is RT decoder */
          val2=8; /* == BX_DEC_CONFIG */
        }
        else if (val2==BX_DEC_CONFIG)
        {
          /* Source is config decoder */
          val2=7; /* == BX_DEC_RT */
        }

        /* Set BX_EGEN_ERR_NUM, i.e. the decoder */
        /* We could activate multiple decoders here, by setting multiple bits !! */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEC_ERR_SRCE_REG,sizeof(bx_int16),1<<val2));
        
        /* Enable decoder errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_SOURCE_REG,sizeof(bx_int16),1<<5));
        break;
      default:
        
        BX_TRY_FAIL(BX_E_INVALID_CASE);/* Invalid property value */
        break;
    }


    /* BX_EGEN_ERR_PHASE */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_PHASE,&val1));
    switch (val1)
    {
      case BX_EGEN_ERR_PHASE_ADDR:
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_PH_EN_REG,sizeof(bx_int16),0));
        break;
      case BX_EGEN_ERR_PHASE_ATTR:
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_PH_EN_REG,sizeof(bx_int16),1));
        break;
      default:
        /* Inject error in data-phase */
        assert(val1>0 && val1 <=1024); /* else: no error would be injected */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_PH_EN_REG,sizeof(bx_int16),2));
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_PH_CNT_REG,sizeof(bx_int16),val1));
        break;
    }

    /* BX_EGEN_WRPAR,BX_EGEN_WRPAR64,BX_EGEN_PERR,BX_EGEN_SERR,BX_EGEN_ERR_ECC,BX_EGEN_ERR_SUBPHASE */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_WRPAR,&val1));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_WRPAR64,&val2));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_PERR,&val3));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_SERR,&val4));
    val5=0;
    val6=0;
    if (!BestXHasMephisto(handle))
    {
      BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_ECC,&val5));       /* 2 bit */
      
      /* Caution:
         BX_EGEN_ERR_SUBPHASE=0 is handled by HW like BX_EGEN_ERR_SUBPHASE=1,
         i.e. error is generated in Subphase one (if BX_EGEN_ERR_ECC is >0) 
      */
      BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ERR_SUBPHASE,&val6));  /* 4 bit */
    }
    assert(val1<2);assert(val2<2);assert(val3<2);assert(val4<2);assert(val5<3);assert(val6<16);
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_EN_REG,sizeof(bx_int16),val1|val2<<1|val3<<2|val4<<3|val5<<4|val6<<6));


    /* BX_EGEN_INT_SOURCE and BX_EGEN_INT_NUM */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_SOURCE,&val1));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_NUM,&val2));

    switch (val1)
    {
      case BX_EGEN_INT_SOURCE_NONE:
        /* No error generation at all */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),0));
        /* BX_EGEN_INT_NUM irrelevant in this case */
        break;
      case BX_EGEN_INT_SOURCE_RIBLK:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_RIGEN_NUMBLK[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error block is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBLK,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_RIGEN_NUMBLK");

        #ifdef HOOK_BESTX_EGEN_PROG_RI_INT_NUM
          HOOK_BESTX_EGEN_PROG_RI_INT_NUM ;
        #endif

        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_INT_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_INT_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable Block errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<0));
        break;
      case BX_EGEN_INT_SOURCE_RIBEH:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_RIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBEH,&val3));

        #ifdef HOOK_BESTX_EGEN_PROG_RI_INT_NUM
          HOOK_BESTX_EGEN_PROG_RI_INT_NUM ;
        #endif

        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_RIGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_INT_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEH_INT_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable master behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<1));
        break;
      case BX_EGEN_INT_SOURCE_CTBEH:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_CTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCTGenGet (handle,BX_CTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_CTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR)); 
        
        /* Set BX_EGEN_INT_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEH_INT_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable target behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<8));
        break;
      case BX_EGEN_INT_SOURCE_CIBEH:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_CIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCIGenGet (handle,BX_CIGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_CIGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_INT_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEH_INT_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable completer behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<2));
        break;
      case BX_EGEN_INT_SOURCE_RTBEH:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_RTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRTGenGet (handle,BX_RTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_RTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_INT_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEH_INT_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable requester behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<3));
        break;
      case BX_EGEN_INT_SOURCE_DEC:
        /*
           If val2 (BX_EGEN_INT_NUM) not in [0,..,BX_DECODER_SIZE[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           decoder is never accessed (because it does not exist).
        */ 
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_INT_NUM should be less than BX_DECODER_SIZE");
        BX_TRY_FAIL((val2<BX_DECODER_SIZE ? BX_E_OK:BX_E_ERROR));
        
        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val2==BX_DEC_RT /* ==7 */)
        {
          /* Source is RT decoder */
          val2=8; /* == BX_DEC_CONFIG */
        }
        else if (val2==BX_DEC_CONFIG)
        {
          /* Source is config decoder */
          val2=7; /* == BX_DEC_RT */
        }

        /* Set BX_EGEN_INT_NUM, i.e. the decoder */
        /* We could activate multiple decoders here, by setting multiple bits !! */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEC_INT_SRCE_REG,sizeof(bx_int16),1<<val2));
        
        /* Enable decoder errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<5));
        break;
      case BX_EGEN_INT_SOURCE_RIDONE:
        if (BestXHasFaust(handle))
        {
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_SOURCE_REG,sizeof(bx_int16),1<<6));
        }
        else
        {
          /* Should never happen (already catched in BestXExerciserGenSet() */
        }
        break;
      default:
        
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid property value */
        break;
    }

    /* BX_EGEN_INT_DELAYA */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_DELAYA,&val1));
    /* Writing a zero here, disables the interrupt generation */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTA_CLK_CNT_REG,sizeof(bx_int16),val1));

    /* BX_EGEN_INT_DELAYB */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_DELAYB,&val1));
    /* Writing a zero here, disables the interrupt generation */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTB_CLK_CNT_REG,sizeof(bx_int16),val1));
   
    /* BX_EGEN_INT_DELAYC */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_DELAYC,&val1));
    /* Writing a zero here, disables the interrupt generation */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTC_CLK_CNT_REG,sizeof(bx_int16),val1));
    
    /* BX_EGEN_INT_DELAYD */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_INT_DELAYD,&val1));
    /* Writing a zero here, disables the interrupt generation */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTD_CLK_CNT_REG,sizeof(bx_int16),val1));

    /* BX_EGEN_TRIG_SOURCE and BX_EGEN_TRIG_NUM */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_TRIG_SOURCE,&val1));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_TRIG_NUM,&val2));

    /***************************************************************************
       chris: clearcase problem with xexer.c: 
       This is what I already had fixed in version xexer.c@@/main/86 */

    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_EN_REG,sizeof(bx_int16),(val1==BX_EGEN_TRIG_SOURCE_NONE?0:1)));

    /***************************************************************************/

    switch (val1)
    {
      case BX_EGEN_TRIG_SOURCE_NONE:
        /* No error generation at all */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),0));
        /* BX_EGEN_TRIG_NUM irrelevant in this case */
        break;
      case BX_EGEN_TRIG_SOURCE_RIBLK:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in [0,..,BX_RIGEN_NUMBLK[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error block is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBLK,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_RIGEN_NUMBLK");

        #ifdef HOOK_BESTX_EGEN_PROG_RI_TRIG_NUM
          HOOK_BESTX_EGEN_PROG_RI_TRIG_NUM ;
        #endif

        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_TRIG_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_TRIG_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable Block errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<0));
        break;
      case BX_EGEN_TRIG_SOURCE_RIBEH:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in [0,..,BX_RIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRIGenGet (handle,BX_RIGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_RIGEN_NUMBEH");

        #ifdef HOOK_BESTX_EGEN_PROG_RI_TRIG_NUM
          HOOK_BESTX_EGEN_PROG_RI_TRIG_NUM ;
        #endif

        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_TRIG_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEH_TRIG_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable master behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<1));
        break;
      case BX_EGEN_TRIG_SOURCE_CTBEH:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in [0,..,BX_CTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCTGenGet (handle,BX_CTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_CTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_TRIG_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEH_TRIG_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable target behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<8));
        break;
      case BX_EGEN_TRIG_SOURCE_CIBEH:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in [0,..,BX_CIGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXCIGenGet (handle,BX_CIGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_CIGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_TRIG_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEH_TRIG_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable completer behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<2));
        break;
      case BX_EGEN_TRIG_SOURCE_RTBEH:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in [0,..,BX_RTGEN_NUMBEH[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           error behavior is never executed.
        */ 
        BX_TRY(BestXRTGenGet (handle,BX_RTGEN_NUMBEH,&val3));
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_RTGEN_NUMBEH");
        BX_TRY_FAIL((val2<val3 ? BX_E_OK:BX_E_ERROR));
        
        /* Set BX_EGEN_TRIG_NUM */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEH_TRIG_SRCE_REG,sizeof(bx_int16),val2));
        
        /* Enable requester behavior errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<3));
        break;
      case BX_EGEN_TRIG_SOURCE_DEC:
        /*
           If val2 (BX_EGEN_TRIG_NUM) not in {0,..,BX_DECODER_SIZE[,
           we return an error here, because in this case,
           no error would be generated at all, because the wanted 
           decoder is never accessed (because it does not exist).
        */ 
        BX_E_ERROR_MSG_SET("BestXExerciserGenProg: BX_EGEN_TRIG_NUM should be less than BX_DECODER_SIZE");
        BX_TRY_FAIL((val2<BX_DECODER_SIZE ? BX_E_OK:BX_E_ERROR));
        
        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val2==BX_DEC_RT /* ==7 */)
        {
          /* Source is RT decoder */
          val2=8; /* == BX_DEC_CONFIG */
        }
        else if (val2==BX_DEC_CONFIG)
        {
          /* Source is config decoder */
          val2=7; /* == BX_DEC_RT */
        }

        /* Set BX_EGEN_TRIG_NUM, i.e. the decoder */
        /* We could activate multiple decoders here, by setting multiple bits !! */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEC_TRIG_SRCE_REG,sizeof(bx_int16),1<<val2));
        
        /* Enable decoder errors, disable all others */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_SOURCE_REG,sizeof(bx_int16),1<<5));
        break;
      default:
        
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid property value */
        break;
    }

    /* BX_EGEN_ARB, BX_EGEN_ARBMASTER, BX_EGEN_ARBCOMPLETER */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ARB,&val1));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ARBRI,&val2));
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_ARBCI,&val3));

    {
      /* Here are the corresponding HW-registers:                   */
      bx_int32 Mode,          /* auto,const,rand,incr               */
               Cnt,           /* clocks to wait until queue skipped */
               RIMin,     /* BX_EGEN_ARBRI                      */
               RIMax,     /* only for mode=increment            */
               CIMin,     /* BX_EGEN_ARBCI                      */
               CIMax;     /* only for mode=increment            */
      
      switch (val1)
      {
        case BX_EGEN_ARB_AUTO:
          /* This is the default */
          Mode=0;           /* Auto/fixed                                    */
          Cnt=0;            /* do not wait, skip immediately, if queue empty */
          RIMin=1;      /* change queue after each xaction               */
          CIMin=1;   /* change queue after each xaction               */
          RIMax=1;      /* dummy, ignored in HW                          */
          CIMax=1;   /* dummy, ignored in HW                          */
          break;

        case BX_EGEN_ARB_CONST:
          /* Take constant number of xactions out of each queue */
          Mode=0;            /* Auto/fixed                      */
          Cnt=0xfffff;       /* wait forever, never skip        */
          RIMin=val2;        /* BX_RIGEN_ARBRI                  */
          CIMin=val3;        /* BX_RIGEN_ARBCI                  */
          RIMax=1;           /* dummy, ignored in HW            */
          CIMax=1;           /* dummy, ignored in HW            */
          break;

        case BX_EGEN_ARB_RAND:
          /* Randomize number of xactions between 1 and 31 */
          Mode=3;       /* random */
          Cnt=1<<17;    /* skip empty queue after waiting this number of clocks */
          RIMin=1;      /* dummy, ignored in HW */
          CIMin=1;      /* dummy, ignored in HW */
          RIMax=1;      /* dummy, ignored in HW */
          CIMax=1;      /* dummy, ignored in HW */
          break;

        case BX_EGEN_ARB_INCR:
          /* Increment number of xactions */
          Mode=2;            /* increment */
          Cnt=1<<17;         /* skip empty queue after waiting this number of clocks */
          RIMin=val2;        /* Startvalue BX_EGEN_ARBRI */
          CIMin=val3;        /* Startvalue BX_EGEN_ARBCI */
          RIMax=255;         /* Endvalue */
          CIMax=255;         /* Endvalue */
          /* HW needs following assertions.
             This is why BX_EGEN_ARBRI and 
             BX_EGEN_ARBCI have to be in range [1,,.254] */
          assert(RIMin<RIMax);
          assert(CIMin<CIMax);
          break;
        
        default:
          
          BX_TRY_FAIL(BX_E_INVALID_CASE);/* Invalid property value */
          break;
      } /* switch */
      
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_XACT_MODE_REG,sizeof(bx_int16),Mode));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_XACT_WAIT_CNT_REG,sizeof(bx_int32),Cnt));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_NUM_M_XACTS_MIN_REG,sizeof(bx_int16),RIMin));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_NUM_M_XACTS_MAX_REG,sizeof(bx_int16),RIMax));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_NUM_C_XACTS_MIN_REG,sizeof(bx_int16),CIMin));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_NUM_C_XACTS_MAX_REG,sizeof(bx_int16),CIMax));

    }


    /* BX_EGEN_DATAGEN */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_DATAGEN,&val1));

    #ifdef HOOK_BESTX_EGEN_PROG_DATAGEN
      HOOK_BESTX_EGEN_PROG_DATAGEN ;
    #endif

    switch (val1)
    {
      case BX_EGEN_DATAGEN_COUNTER:
        val2=3;
        break;
      case BX_EGEN_DATAGEN_WALKING1:
        val2=0;
        break;
      case BX_EGEN_DATAGEN_WALKING0:
        val2=1;
        break;
      case BX_EGEN_DATAGEN_COUNTMIX:
        val2=4;
        break;
      case BX_EGEN_DATAGEN_GROUNDBOUNCE:
        val2=2;
        break;
      default:
        
        BX_TRY_FAIL(BX_E_INVALID_CASE);/* Invalid property value */
        break;
    }

    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MDGEN_PATRNSEL_REG,sizeof(bx_int16),val2));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TDGEN_PATRNSEL_REG,sizeof(bx_int16),val2));

    /* BX_EGEN_DATAFIX */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_DATAFIX,&val1));
    if (val1==BX_EGEN_DATAFIX_MASTERID)
    {
      /* This is the default */
      /* Read initiator id from configuration space
         (PCI-X status register, bits 0-15) */
      
      BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+0x64,sizeof(bx_int32),&val2));
      /* Extract bits 0-15 (i.e. initiator ID) */
      val2 &= 0xffff;
    }
    else
    {
      /* Take 18-bit value from user */
      val2=val1;      
    }
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MDGEN_ID_REG,sizeof(bx_int32),val2));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TDGEN_ID_REG,sizeof(bx_int32),val2));


    /* BX_EGEN_DATASEED */
    BX_TRY(BestXExerciserGenGet (handle,BX_EGEN_DATASEED,&val1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MDGEN_OFFSET_REG,sizeof(bx_int32),val1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TDGEN_OFFSET_REG,sizeof(bx_int32),val1));
     

    /* BX_EGEN_PARTCOMP */
    BX_TRY(BestXExerciserGenGet(handle,BX_EGEN_PARTCOMP,&val1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_CTRL_REG,sizeof(bx_int16),val1));

    /* Switch to run mode */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));

    #ifdef HOOK_BESTX_EGEN_PROG_END
      HOOK_BESTX_EGEN_PROG_END ;
    #endif

  }
  
  BX_TRY_CATCH
  {
    /* error occurred */
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* Switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void)BestXExerciserRunMode(handle,1 /* do soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between  returned an error */
        (void)BestXExerciserRunMode(handle,1 /* do soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  } /* CATCH */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserGenRead
 *
 * Purpose: Reads all generic properties from card- to host-memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXExerciserGenRead(bx_handletype  handle)
{
  /* Updates generics memory on host DB */

  BX_DECLARE_FUNCNAME("BestXExerciserGenRead [egenread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 val1,val2,val3;

    /* Switch to prog mode */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
    
    /* BX_EGEN_ERR_SOURCE and BX_EGEN_ERR_NUM */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_ERR_SOURCE_REG, sizeof(bx_int16),&val1));
    switch (val1)
    { 
      case 1<<0: /* RIBLK */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_RIBLK));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBLOCK_ERR_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_ERR_NUM
          HOOK_BESTX_EGEN_READ_RI_ERR_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val2));
        break;
      case 1<<1: /* RIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_RIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBEH_ERR_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_ERR_NUM
          HOOK_BESTX_EGEN_READ_RI_ERR_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val2));
        break;
      case 1<<4: /* CTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_CTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TBEH_ERR_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val2));
        break;
      case 1<<2: /* CIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_CIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_CBEH_ERR_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val2));
        break;
      case 1<<3: /* RTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_RTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_RTBEH_ERR_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val2));
        break;
      case 1<<5: /* DEC */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_DEC));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_ERR_SRCE_REG,sizeof(bx_int16),&val2));
        /* Compute the first bit set in val2. 
           This is the decoder number of bx_dectype (exception: RT<->CFG).
           There should be exactly one bit set !! */
        val3=0; 
        while (val2 && val2&1==0) 
        {
          val3++;
          val2=val2>>1;
        }
        assert(val3<BX_DECODER_SIZE);

        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val3==7 /* ==BX_DEC_RT */)
        {
          /* Source is config decoder */
          val3=BX_DEC_CONFIG;  /* ==8 */
        }
        else if (val3==8 /* ==BX_DEC_CONFIG */)
        {
          /* Source is RT decoder */
          val3=BX_DEC_RT; /* == 7 */
        }

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,val3));
        break;
      default:
        /* HW-register should never contain this value, 
           maybe multiple bits have been set manually ?? 
           We map this case to NONE 
        */
      case 0: /* NONE */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_SOURCE,BX_EGEN_ERR_SOURCE_NONE));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_NUM,0)); /* irrelevant */
        break;
    }
    
    /* BX_EGEN_ERR_PHASE */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_PH_EN_REG,sizeof(bx_int16),&val1));
    switch (val1)
    {
      case 0:
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_PHASE,BX_EGEN_ERR_PHASE_ADDR));
        break;
      case 1:
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_PHASE,BX_EGEN_ERR_PHASE_ATTR));
        break;
      case 2:
      case 3:
        /* Inject error in data-phase val2 */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_PH_CNT_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_ERR_PHASE,val2));
        /* if (val2==0 || val2 > 1024) no error would be injected !! */
        break;
      default:
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* HW-register should never contain this value */
        break;
    }

    /* BX_EGEN_ERR_WRPAR,BX_EGEN_ERR_WRPAR64,BX_EGEN_ERR_PERR,BX_EGEN_ERR_SERR,BX_EGEN_ERR_ECC,BX_EGEN_ERR_SUBPHASE */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_EN_REG,sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_WRPAR,(val1&1?1:0)));
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_WRPAR64,(val1&2?1:0)));
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_PERR,(val1&4?1:0)));
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_SERR,(val1&8?1:0)));
    if (!BestXHasMephisto(handle))
    {
      BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_ECC,(val1>>4)&3));


      /* Caution:
         BX_EGEN_ERR_SUBPHASE=0 is handled by HW like BX_EGEN_ERR_SUBPHASE=1,
         i.e. error is generated in Subphase one (if BX_EGEN_ERR_ECC is >0) 
      */
      BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ERR_SUBPHASE,(val1>>6)&0xf));
    }

    /* BX_EGEN_INT_SOURCE and BX_EGEN_INT_NUM */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_INT_SOURCE_REG, sizeof(bx_int16),&val1));
    switch (val1)
    { 
      case 1<<0: /* RIBLK */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_RIBLK));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBLOCK_INT_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_INT_NUM
          HOOK_BESTX_EGEN_READ_RI_INT_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val2));
        break;
      case 1<<1: /* RIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_RIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBEH_INT_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_INT_NUM
          HOOK_BESTX_EGEN_READ_RI_INT_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val2));
        break;
      case 1<<4: /* CTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_CTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TBEH_INT_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val2));
        break;
      case 1<<2: /* CIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_CIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_CBEH_INT_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val2));
        break;
      case 1<<3: /* RTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_RTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_RTBEH_INT_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val2));
        break;
      case 1<<5: /* DEC */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_DEC));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_INT_SRCE_REG,sizeof(bx_int16),&val2));
        /* Compute the first bit set in val2. This is the decoder number of bx_dectype.
           There should be exactly one bit set !! */
        val3=0; 
        while (val2 && val2&1==0) 
        {
          val3++;
          val2=val2>>1;
        }
        assert(val3<BX_DECODER_SIZE);
        
        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val3==7 /* ==BX_DEC_RT */)
        {
          /* Source is config decoder */
          val3=BX_DEC_CONFIG;  /* ==8 */
        }
        else if (val3==8 /* ==BX_DEC_CONFIG */)
        {
          /* Source is RT decoder */
          val3=BX_DEC_RT; /* == 7 */
        }

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,val3));
        break;
      case 1<<6: /* RIDONE */
        if (BestXHasFaust(handle))
        {
          BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_RIDONE));
        }
        else
        {
          /* BX_EGEN_INT_SOURCE_RIDONE not supported on Mephisto HW */
          BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_NONE));
        }
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,0)); /* irrelevant */
        break;
      default:
        /* HW-register should never contain this value, 
           maybe multiple bits have been set manually ?? 
           We map this case to NONE 
        */
      case 0: /* NONE */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_SOURCE,BX_EGEN_INT_SOURCE_NONE));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_NUM,0)); /* irrelevant */
        break;
    }
    

    /* BX_EGEN_INT_DELAYA */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_INTA_CLK_CNT_REG, sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_DELAYA,val1));

    /* BX_EGEN_INT_DELAYB */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_INTB_CLK_CNT_REG, sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_DELAYB,val1));
    
    /* BX_EGEN_INT_DELAYC */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_INTC_CLK_CNT_REG, sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_DELAYC,val1));
    
    /* BX_EGEN_INT_DELAYD */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_INTD_CLK_CNT_REG, sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_INT_DELAYD,val1));

    /* BX_EGEN_TRIG_SOURCE and BX_EGEN_TRIG_NUM */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_TRIG_SOURCE_REG, sizeof(bx_int16),&val1));
    switch (val1)
    { 
      case 1<<0: /* RIBLK */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_RIBLK));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBLOCK_TRIG_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_TRIG_NUM
          HOOK_BESTX_EGEN_READ_RI_TRIG_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val2));
        break;
      case 1<<1: /* RIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_RIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MBEH_TRIG_SRCE_REG,sizeof(bx_int16),&val2));

        #ifdef HOOK_BESTX_EGEN_READ_RI_TRIG_NUM
          HOOK_BESTX_EGEN_READ_RI_TRIG_NUM ;
        #endif

        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val2));
        break;
      case 1<<4: /* CTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_CTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TBEH_TRIG_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val2));
        break;
      case 1<<2: /* CIBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_CIBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_CBEH_TRIG_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val2));
        break;
      case 1<<3: /* RTBEH */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_RTBEH));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_RTBEH_TRIG_SRCE_REG,sizeof(bx_int16),&val2));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val2));
        break;
      case 1<<5: /* DEC */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_DEC));
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_TRIG_SRCE_REG,sizeof(bx_int16),&val2));
        /* Compute the first bit set in val2. This is the decoder number of bx_dectype.
           There should be exactly one bit set !! */
        val3=0; 
        while (val2 && val2&1==0) 
        {
          val3++;
          val2=val2>>1;
        }
        assert(val3<BX_DECODER_SIZE);

        /* Bitpositions 7 and 8 (for rt and cfg decoder) are swapped in register BX_REG_DEC_ERR_SRCE_REG
           when compared to the decoder enable register resp. the bx_dectype in CAPI. 
           In order to be able to use bx_dectype as parameter, we swap it here: 
        */
        if (val3==7 /* ==BX_DEC_RT */)
        {
          /* Source is config decoder */
          val3=BX_DEC_CONFIG;  /* ==8 */
        }
        else if (val3==8 /* ==BX_DEC_CONFIG */)
        {
          /* Source is RT decoder */
          val3=BX_DEC_RT; /* == 7 */
        }
        
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,val3));
        
        break;
      default:
        /* HW-register should never contain this value, 
           maybe multiple bits have been set manually ?? 
           We map this case to NONE 
        */
      case 0: /* NONE */
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_SOURCE,BX_EGEN_TRIG_SOURCE_NONE));
        BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_TRIG_NUM,0)); /* irrelevant */
        break;
    }
    

    /* BX_EGEN_ARB, BX_EGEN_ARBMASTER, BX_EGEN_ARBCOMPLETER */
    {
      /* Here are the corresponding HW-registers:                   */
      bx_int32 Mode,          /* auto,const,rand,incr               */
               Cnt,           /* clocks to wait until queue skipped */
               RIMin,         /* BX_EGEN_ARBRI                      */
               CIMin;         /* BX_EGEN_ARBCI                      */

      BX_TRY(BestXDirectRegRead(handle,BX_REG_XACT_MODE_REG,sizeof(bx_int16),&Mode));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_XACT_WAIT_CNT_REG,sizeof(bx_int32),&Cnt));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_NUM_M_XACTS_MIN_REG,sizeof(bx_int16),&RIMin));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_NUM_C_XACTS_MIN_REG,sizeof(bx_int16),&CIMin));

      switch (Mode)
      {
        case 0: /* CONST/AUTO mode */
          if (Cnt==0xfffff)
          {
            /* CONST mode */
            val1=BX_EGEN_ARB_CONST;
            val2=RIMin;
            val3=CIMin;
          }
          else
          {
            /* AUTO mode */
            assert(Cnt==0);
            val1=BX_EGEN_ARB_AUTO;
            val2=1;
            val3=1;
          }
          break;

        case 2: /* INCREMENT mode */
          val1=BX_EGEN_ARB_INCR;
          val2=1;
          val3=1;
          break;

        case 3: /* RANDOM mode */
          val1=BX_EGEN_ARB_RAND;
          val2=1;
          val3=1;
          break;

        default:
          
          BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid register value */
          break;
      } /* switch */

      BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ARB,val1));
      BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ARBRI,val2));
      BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_ARBCI,val3));
    }   

    /* BX_EGEN_DATAGEN */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MDGEN_PATRNSEL_REG,sizeof(bx_int16),&val1));
    /* BX_TRY(BestXDirectRegRead(handle,BX_REG_TDGEN_PATRNSEL_REG,sizeof(bx_int16),&val1)); */
    switch (val1)
    {
      case 0x3:
        val2=BX_EGEN_DATAGEN_COUNTER;
        break;
      case 0x0:
        val2=BX_EGEN_DATAGEN_WALKING1;
        break;
      case 0x1:
        val2=BX_EGEN_DATAGEN_WALKING0;
        break;
      case 0x4:
        val2=BX_EGEN_DATAGEN_COUNTMIX;
        break;
      case 0x2:
        val2=BX_EGEN_DATAGEN_GROUNDBOUNCE;
        break;
      default:
        
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid register value */
        break;
    }
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_DATAGEN,val2));

    /* BX_EGEN_DATAFIX */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MDGEN_ID_REG,sizeof(bx_int32),&val1));
    /* BX_TRY(BestXDirectRegRead(handle,BX_REG_TDGEN_ID_REG,sizeof(bx_int32),&val1)); */
    /* Problem: Card does only store a value, but does not know
       wether this value is meant to be the initiator id or not.
       Two possible solutions:
       1. Compare the stored value with the initiator id.
          If equal, set BX_EGEN_DATAFIX to BX_EGEN_DATAFIX_MASTERID.
          This means card will usually not startup with the default value.
       2. Never set BX_EGEN_DATAFIX to BX_EGEN_DATAFIX_MASTERID
          This means card does never startup with the default value
          BX_EGEN_DATAFIX_MASTERID.*/
    
    /* Implement the second solution: */
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_DATAFIX,val1));

    /* BX_EGEN_DATASEED */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MDGEN_OFFSET_REG,sizeof(bx_int32),&val1));
    /* BX_TRY(BestXDirectRegRead(handle,BX_REG_TDGEN_OFFSET_REG,sizeof(bx_int32),&val1)); */
    BX_TRY(BestXExerciserGenSet (handle,BX_EGEN_DATASEED,val1));
    
    #ifdef HOOK_BESTX_EGEN_READ_DATAGEN
      HOOK_BESTX_EGEN_READ_DATAGEN ;
    #endif

    /* BX_EGEN_PARTCOMP */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CTRL_REG,sizeof(bx_int16),&val1));
    BX_TRY(BestXExerciserGenSet(handle,BX_EGEN_PARTCOMP,val1));

    /* Switch to run mode */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    /* error occurred */
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* Switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void)BestXExerciserRunMode(handle,1 /* do soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between  returned an error */
        (void)BestXExerciserRunMode(handle,1 /* do soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  } /* CATCH */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXExerciserDefaultSet
 *
 * Purpose: Defaults all exerciser props in DB. 
 *          Marks memories on card to be up to date.
 * Precondition: An allocated and initialized DB must already be present.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXExerciserDefaultSet (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXExerciserDefaultSet [edefset]");

  bx_int32 mask,val,i,ConfigSize;
  

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);

    /* Bring decoders in config space to a sensible default state */
    BX_TRY(BestXTDecoderDefaultSet(handle));

    /* Default config space */
    for (i=0;i<ConfigSize;i++)
    {
      BX_TRY(BestXConfRegDefaultGet(handle,i*4,&val));
      BX_TRY(BestXConfRegMaskDefaultGet(handle,i*4,&mask));
      bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.mask [i] = mask;
      bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.value[i] = val;
    }

    /* The latency timer's value in config space is mode dependent:
       0x00 in PCI and 0x40 in PCIX mode 
    */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_PCIX_MEN_REG,sizeof(bx_int16),&val));
    if (val&4)
    {
      /* Card is in PCIX mode, set bit */
      bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.value[3] |= 0x4000;
    }
    else
    {
      /* Card is in PCI mode, clear bit */
      bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.value[3] &= ~0x4000;
    }

#ifdef CUSTOM_OEM1
/* For Intel, we don't want the apps to accidentially
   set the decoders and modes to library defaults.
   We have our own set of PU defaults and setting them
   to lib defaults here will break us bad.  Since most
   of our apps will start by setting the exerciser to
   defaults, we will just read the current values from
   the card.
*/
    BX_TRY(BestXTDecoderRead(handle));
    BX_TRY(BestXTConfigRead(handle));
#endif

    /* Generic properties */
    BX_TRY(BestXRIGenDefaultSet(handle));
    BX_TRY(BestXCTGenDefaultSet(handle));
    BX_TRY(BestXCIGenDefaultSet(handle));
    BX_TRY(BestXRTGenDefaultSet(handle));
    BX_TRY(BestXExerciserGenDefaultSet(handle));

    /* Split conditions */
    BX_TRY(BestXCTSplitCondDefaultSet(handle,0));
    BX_TRY(BestXCTSplitCondDefaultSet(handle,1));
    BX_TRY(BestXCTSplitCondDefaultSet(handle,2));
    BX_TRY(BestXCTSplitCondDefaultSet(handle,3));

    /* Memories (Block + Behavior) */
    BX_TRY(BestXRIBlockMemInit(handle));
    BX_TRY(BestXRIBehMemInit(handle));
    BX_TRY(BestXCTBehMemInit(handle));
    BX_TRY(BestXCIBehMemInit(handle));
    BX_TRY(BestXRTBehMemInit(handle));
       
    /* Mark memories to be valid, so only
       changes are written down to card
    */

    SetUpdateInfo(RIBLK_UPD,0,0,0); 
    SetUpdateInfo(RIBEH_UPD,0,0,0); 
    SetUpdateInfo(CTBEH_UPD,0,0,0); 
    SetUpdateInfo(CIBEH_UPD,0,0,0); 
    SetUpdateInfo(RTBEH_UPD,0,0,0); 

    /* Mark first memory line to be invalid,
       so a following ExerciserProg() will at least
       program a default behavior.
    */

    BX_TRY(BestXRIBlockDefaultSet(handle,0));
    BX_TRY(BestXRIBehDefaultSet(handle,0));
    BX_TRY(BestXCTBehDefaultSet(handle,0));
    BX_TRY(BestXCIBehDefaultSet(handle,0));
    BX_TRY(BestXRTBehDefaultSet(handle,0));
  }

  BX_ERRETURN(BX_TRY_RET);
}
